'use client';

import Image from 'next/image';
import { useState } from 'react';
import type { Product, Addon } from '@/lib/types';
import { useCart } from '@/hooks/use-cart';
import { useToast } from '@/hooks/use-toast';
import { Button } from '@/components/ui/button';
import {
  Card,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from '@/components/ui/card';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogClose,
} from '@/components/ui/dialog';
import { Checkbox } from '@/components/ui/checkbox';
import { Label } from '@/components/ui/label';
import { Plus } from 'lucide-react';

interface ProductCardProps {
  product: Product;
}

const formatPrice = (price: number) => {
  return price.toLocaleString('pt-BR', {
    style: 'currency',
    currency: 'BRL',
  });
};

export function ProductCard({ product }: ProductCardProps) {
  const { addItem } = useCart();
  const { toast } = useToast();
  
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [selectedAddons, setSelectedAddons] = useState<Addon[]>([]);

  const handleAddItem = () => {
    if (product.addons && product.addons.length > 0) {
      setIsDialogOpen(true);
    } else {
      addItem(product, [], 1);
      toast({
        title: 'Adicionado!',
        description: `${product.name} foi adicionado à sua sacola.`,
      });
    }
  };

  const handleAddonToggle = (addon: Addon, checked: boolean) => {
    setSelectedAddons(prev => 
      checked ? [...prev, addon] : prev.filter(a => a.id !== addon.id)
    );
  };

  const handleConfirmAdd = () => {
    addItem(product, selectedAddons, 1);
    toast({
      title: 'Adicionado!',
      description: `${product.name} com adicionais foi adicionado à sua sacola.`,
    });
    setIsDialogOpen(false);
    setSelectedAddons([]);
  }

  const calculateTotalPrice = () => {
    const addonsPrice = selectedAddons.reduce((total, addon) => total + addon.price, 0);
    return product.price + addonsPrice;
  }

  return (
    <>
      <Card className="flex flex-col overflow-hidden shadow-md hover:shadow-xl transition-shadow duration-300">
        {product.imageUrl && (
          <div className="relative w-full h-48">
            <Image
              src={product.imageUrl}
              alt={product.name}
              fill
              className="object-cover"
            />
          </div>
        )}
        <CardHeader>
          <CardTitle>{product.name}</CardTitle>
          <CardDescription className="flex-grow min-h-[40px]">
            {product.description}
          </CardDescription>
        </CardHeader>
        <CardContent className="flex-grow" />
        <CardFooter className="flex justify-between items-center">
          <p className="text-xl font-bold text-primary">
            {formatPrice(product.price)}
          </p>
          <Button onClick={handleAddItem} aria-label={`Adicionar ${product.name} ao carrinho`}>
            <Plus className="h-4 w-4 mr-2" />
            Adicionar
          </Button>
        </CardFooter>
      </Card>
      
      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Adicionais para {product.name}</DialogTitle>
          </DialogHeader>
          <div className="space-y-4 py-4">
            {product.addons?.map(addon => (
              <div key={addon.id} className="flex items-center space-x-3">
                <Checkbox 
                  id={`addon-${product.id}-${addon.id}`}
                  onCheckedChange={(checked) => handleAddonToggle(addon, checked as boolean)}
                />
                <Label 
                  htmlFor={`addon-${product.id}-${addon.id}`}
                  className="flex justify-between w-full font-normal"
                >
                  <span>{addon.name}</span>
                  <span className="text-muted-foreground">{`+ ${formatPrice(addon.price)}`}</span>
                </Label>
              </div>
            ))}
          </div>
          <DialogFooter>
            <div className="w-full flex justify-between items-center">
                <span className="text-lg font-bold">Total: {formatPrice(calculateTotalPrice())}</span>
                <div>
                    <DialogClose asChild>
                        <Button variant="ghost">Cancelar</Button>
                    </DialogClose>
                    <Button onClick={handleConfirmAdd}>Confirmar</Button>
                </div>
            </div>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </>
  );
}
